<?php

// Direct test with user's card using 3D Secure
require_once __DIR__ . '/vendor/autoload.php';

use Iyzipay\Options;
use Iyzipay\Model\ThreedsInitialize;
use Iyzipay\Request\CreateThreedsInitializeRequest;
use Iyzipay\Model\Buyer;
use Iyzipay\Model\Address;
use Iyzipay\Model\BasketItem;
use Iyzipay\Model\PaymentCard;

echo "=== Direkt 3D Secure Test - Kullanıcı Kartı ===\n\n";

try {
    // Load config
    $config = require __DIR__ . '/config/iyzico.php';
    
    echo "1. API Bilgileri:\n";
    echo "   API Key: " . substr($config['api_key'], 0, 8) . "...\n";
    echo "   Secret Key: " . substr($config['secret_key'], 0, 8) . "...\n";
    echo "   Keys identical: " . ($config['api_key'] === $config['secret_key'] ? 'EVET (Bu sorun olabilir!)' : 'Hayır') . "\n\n";
    
    // Initialize options
    $options = new Options();
    $options->setApiKey($config['api_key']);
    $options->setSecretKey($config['secret_key']);
    $options->setBaseUrl($config['base_url']);
    
    echo "2. Kart bilgileri:\n";
    echo "   Kart No: 6587****9656\n";
    echo "   Son Kullanma: 12/2025\n";
    echo "   CVC: 152\n\n";
    
    // Create 3D Secure request
    $request = new CreateThreedsInitializeRequest();
    $request->setLocale(\Iyzipay\Model\Locale::TR);
    $request->setConversationId('test-3ds-' . time());
    $request->setPrice('10.0');
    $request->setPaidPrice('10.0');
    $request->setCurrency(\Iyzipay\Model\Currency::TL);
    $request->setInstallment(1);
    $request->setBasketId('test-basket-' . time());
    $request->setPaymentChannel(\Iyzipay\Model\PaymentChannel::WEB);
    $request->setPaymentGroup(\Iyzipay\Model\PaymentGroup::PRODUCT);
    $request->setCallbackUrl('https://www.merchant.com/callback');

    // Payment Card - User's card
    $paymentCard = new PaymentCard();
    $paymentCard->setCardHolderName('Test User');
    $paymentCard->setCardNumber('6587619825149656');
    $paymentCard->setExpireMonth('12');
    $paymentCard->setExpireYear('2025');
    $paymentCard->setCvc('152');
    $paymentCard->setRegisterCard(0);
    $request->setPaymentCard($paymentCard);

    // Buyer Information
    $buyer = new Buyer();
    $buyer->setId('test-buyer-' . time());
    $buyer->setName('Test');
    $buyer->setSurname('User');
    $buyer->setGsmNumber('+905350000000');
    $buyer->setEmail('test@example.com');
    $buyer->setIdentityNumber('74300864791');
    $buyer->setLastLoginDate(date('Y-m-d H:i:s', strtotime('-1 hour')));
    $buyer->setRegistrationDate(date('Y-m-d H:i:s', strtotime('-1 year')));
    $buyer->setRegistrationAddress('Test Address Istanbul');
    $buyer->setIp('127.0.0.1');
    $buyer->setCity('Istanbul');
    $buyer->setCountry('Turkey');
    $buyer->setZipCode('34000');
    $request->setBuyer($buyer);

    // Shipping Address
    $shippingAddress = new Address();
    $shippingAddress->setContactName('Test User');
    $shippingAddress->setCity('Istanbul');
    $shippingAddress->setCountry('Turkey');
    $shippingAddress->setAddress('Test Shipping Address');
    $shippingAddress->setZipCode('34000');
    $request->setShippingAddress($shippingAddress);

    // Billing Address
    $billingAddress = new Address();
    $billingAddress->setContactName('Test User');
    $billingAddress->setCity('Istanbul');
    $billingAddress->setCountry('Turkey');
    $billingAddress->setAddress('Test Billing Address');
    $billingAddress->setZipCode('34000');
    $request->setBillingAddress($billingAddress);

    // Basket Items
    $basketItem = new BasketItem();
    $basketItem->setId('test-item-1');
    $basketItem->setName('Test Ürün');
    $basketItem->setCategory1('Test Kategori');
    $basketItem->setCategory2('Alt Kategori');
    $basketItem->setItemType(\Iyzipay\Model\BasketItemType::PHYSICAL);
    $basketItem->setPrice('10.0');
    $request->setBasketItems([$basketItem]);

    echo "3. 3D Secure ödeme başlatılıyor...\n";
    
    $threedsInitialize = ThreedsInitialize::create($request, $options);

    echo "4. Sonuç:\n";
    echo "   Status: " . $threedsInitialize->getStatus() . "\n";
    
    if ($threedsInitialize->getStatus() == 'success') {
        echo "✓ 3D Secure başlatma başarılı!\n";
        echo "   Payment ID: " . $threedsInitialize->getPaymentId() . "\n";
        echo "   Conversation ID: " . $threedsInitialize->getConversationId() . "\n";
        echo "   3D Secure HTML uzunluğu: " . strlen($threedsInitialize->getHtmlContent()) . " karakter\n\n";
        
        // Save HTML content
        file_put_contents(__DIR__ . '/3d_secure_form.html', $threedsInitialize->getHtmlContent());
        echo "✓ 3D Secure formu '3d_secure_form.html' dosyasına kaydedildi\n";
        echo "Bu dosyayı tarayıcıda açarak test edebilirsiniz\n\n";
        
        echo "Mobil entegrasyon için:\n";
        echo "- Bu HTML içeriğini WebView'da gösterin\n";
        echo "- Kullanıcı 3D Secure doğrulamasını tamamlayacak\n";
        echo "- Başarılı olursa callback URL'e yönlendirilecek\n";
        
    } else {
        echo "✗ 3D Secure başlatma başarısız:\n";
        echo "   Error: " . $threedsInitialize->getErrorMessage() . "\n";
        echo "   Error Code: " . $threedsInitialize->getErrorCode() . "\n";
        
        if ($threedsInitialize->getErrorCode() == '1001') {
            echo "\n⚠️  Hata 1001: API bilgileri geçersiz\n";
            echo "   - API Key ve Secret Key'in doğru olduğundan emin olun\n";
            echo "   - Sandbox/Production ortamının doğru olduğunu kontrol edin\n";
            echo "   - Iyzico panelinden API bilgilerini yeniden kontrol edin\n";
        }
    }
    
} catch (Exception $e) {
    echo "✗ Exception: " . $e->getMessage() . "\n";
    echo "File: " . $e->getFile() . ":" . $e->getLine() . "\n";
}

echo "\n=== Test Tamamlandı ===\n";
